/********** 
  ILPS (Iterated Local & Plateau Search) for
  the Minimum Independent Dominating Set Problem

  main.cpp

  Copyright 2018 Kazuya Haraguchi
  Released Under the MIT License
  https://opensource.org/licenses/mit-license.php
**********/

#include "define.h"
#include "init.h"
#include "construct.h"
#include "misc.h"

void ILPS(ParamSet PS, Graph G, Solution Best, Tool T);
bool plateauSearch(ParamSet PS, Graph G, Solution S, Tool T);
bool localSearchII(Graph G, Solution S, Tool T);
bool localSearchIII(ParamSet PS, Graph G, Solution S, Tool T);
bool localSearchIII_3TightBase(Graph G, Solution S, Tool T,
			       VertexProxy u, VertexProxy X[3]);
bool localSearchIII_2TightBase(Graph G, Solution S, Tool T,
			       VertexProxy U[2], VertexProxy X[3], bool adj,
			       Seq *T1N, Seq &NX_2, BinTree &NXP_2, Seq &NXP_3);
void updatePenalty(ParamSet PS, Graph G, Solution S, Tool T);
void forget(ParamSet PS, Graph G, Solution S, Tool T);
void kick(ParamSet PS, Graph G, Solution Best, Solution S, Tool T);

/*** main function ***/
int main(int argc, char *argv[]){
  ParamSet PS;
  Graph G;
  Solution Best;
  Tool T;
  int i;
  
  /* preprocess */
  PS = new struct _ParamSet;
  G = new struct _Graph;
  Best = new struct _Solution;
  T = new struct _Tool;

  readParam(argc, argv, PS);
  readInstance(stdin, G);
  if(PS->itrmax<0)
    PS->itrmax = -(G->n*PS->itrmax);
  initSolution(G, Best);
  initTool(G, T);
  init_genrand(PS->seed);
  cpu_time();

#ifdef DEBUG
  outputGraph(stdout, G);
#else
  printf("NUM_VERTICES:\t%d\n", G->n);
  printf("NUM_EDGES:\t%d\n", G->m);
#endif

  /* construct the initial solution */
  if(PS->init == 0)
    constructRandomSol(G, Best);
  else
    constructGreedySol(G, Best);

#ifdef DEBUG
  outputSolution(stdout, Best);
#endif

  /* iterated local & plateau search */
  ILPS(PS, G, Best, T);

  /* postprocess */
  if(isFeasible(G, Best))
    printf("FEASIBILITY_CHECKED\n");
  else{    
    fprintf(stderr, "error: the obtained solution is not IDS.\n");
    outputSolution(stdout, Best);
    exit(EXIT_FAILURE);
  }

  printf("FOUND_SOLUTION:\t");
  bool firstflag = true;
  for(i=0;i<G->n;i++)
    if(Best->X[i]->isSol){
      if(firstflag==false)
	printf(",");
      printf("%d", i+1);
      firstflag = false;
    }
  printf("\n");
  printf("SOLUTION_SIZE:\t%d\n\n", Best->num[SECT_SOL]);

  delete PS;
  delete G;
  delete Best;
  delete T;
  
  return EXIT_SUCCESS;
}


/***** ILPS: iterated local & plateau search *****/
void ILPS(ParamSet PS, Graph G, Solution Best, Tool T){
  Solution S;
  unsigned int threshold;
  threshold = 1 << 31;
  threshold--;

  T->start_time = cpu_time();
  T->update = false;

  S = new struct _Solution;
  initSolution(G, S);
  copySolution(G, Best, S);
  updatePenalty(PS, G, S, T);
  printf("%d\t0\tINIT\n", Best->num[SECT_SOL]);
  
  while(T->t<PS->itrmax){
    if(PS->time>0. && cpu_time()-T->start_time > PS->time){
      printf("ABORT_BY_TIMELIMIT\n");
      break;
    }

    /* local search */
    if(localSearchII(G, S, T))
      continue;
#ifdef LS3
    if(localSearchIII(PS, G, S, T))
      continue;
#endif    
    
    /* plateau search */
#ifdef LS3
    if(S->num[SECT_SOL] <= Best->num[SECT_SOL]+2)
      while(1){
	if(plateauSearch(PS, G, S, T)==false)
	  break;
      }
#else
    while(1){
      if(plateauSearch(PS, G, S, T)==false)
	break;
    }
#endif
    
    /* update the incumbent solution */
    if(S->num[SECT_SOL] <= Best->num[SECT_SOL]){
      if(S->num[SECT_SOL] < Best->num[SECT_SOL]){
	T->itr_tb = T->t;
	T->time_tb = cpu_time();
	printf("%d\t%d\t%g\n", S->num[SECT_SOL], T->itr_tb, T->time_tb-T->start_time);
	// shuffle the random arrays
	for(int i=0;i<G->n;i++)
	  shuffle(G->V[i]->R, G->V[i]->deg, sizeof(int));
      }
      T->update = true;
      copySolution(G, S, Best);
    }

    /* kick */
    kick(PS, G, Best, S, T);
    T->t++;

    /* update penalty */
    updatePenalty(PS, G, S, T);
    forget(PS, G, S, T);
    T->update = false;
    T->a = T->a&threshold; // in case T->a is too large
  }
  if(T->t==PS->itrmax)
    printf("ABORT_BY_ITERATION_TIMES\n");

  delete S;

  printf("ITR_TO_BEST:\t%d\n", T->itr_tb);
  if(T->itr_tb==0)
    printf("TIME_TO_BEST:\t0\n");
  else
    printf("TIME_TO_BEST:\t%g\n", T->time_tb - T->start_time);
  printf("ITRTIMES:\t%d\n", T->t);
  printf("CPU_TIME:\t%g\n", cpu_time()-T->start_time);
}


bool plateauSearch(ParamSet PS, Graph G, Solution S, Tool T){
  Seq *T1N,Drop,Add;
  
  /* construct T1N */
  T1N = new Seq[S->num[SECT_SOL]];
  for(int i=0;i<S->num[SECT_SOL];i++){
    VertexProxy proxy = S->Order[SECT_SOL][i];
    Vertex v = proxy->v;
    for(int r=0;r<v->deg;r++){
      int j = v->AL[ v->R[(T->a++)%v->deg] ];
      if(S->X[j]->tightness==1)
	T1N[i].push_back(j);
    }
  }

  for(int r=0;r<S->num[SECT_SOL];r++){
    int i = (T->a++)%S->num[SECT_SOL];
    VertexProxy solproxy = S->Order[SECT_SOL][i];
    int deg_1tight = T1N[i].size();
    for(auto itr=T1N[i].begin(); itr!=T1N[i].end(); ++itr){
      int j = *itr;
      setNeighborsCounters(S, S->X[j], 0);
      for(auto nb=solproxy->v->AL.begin(); nb!=solproxy->v->AL.end(); ++nb){
	int k = *nb;
	if(S->X[k]->tightness==1 && S->X[k]->isSol==false)
	  S->X[k]->counter += 1;
      }
      int cnt = 0;
      for(auto nb=S->X[j]->v->AL.begin(); nb!=S->X[j]->v->AL.end(); ++nb){
	int k = *nb;
	if(S->X[k]->counter==1)
	  cnt++;
      }
      if(cnt==deg_1tight-1){
	Drop.push_back(solproxy->v->id);
	Add.push_back(S->X[j]->v->id);
      }
    }
  }

  delete[] T1N;
  
  int m=Drop.size();
  if(m==0)
    return false;
  
  for(int k=0;k<m;k++){
    drop(S, S->X[Drop[k]]->order);
    add(S, S->X[Add[k]]->order);
    bool flag = false;
    while(1){
      if(localSearchII(G, S, T)){
	flag = true;
	continue;
      }
#ifdef LS3
      if(localSearchIII(PS, G, S, T)){
	flag = true;
	continue;
      }
#endif
      break;
    }
    if(flag == false){
      drop(S, S->X[Add[k]]->order);
      add(S, S->X[Drop[k]]->order);
    }
    else
      return true;
  }
  int k = genrand_real2() * (double)(m+1);
  if(k<m){
    drop(S, S->X[Drop[k]]->order);
    add(S, S->X[Add[k]]->order);
  }
  return false;
}


bool localSearchII(Graph G, Solution S, Tool T){
  VertexProxy u,v,X[2];
  bool flag=false;
  int i,j,bi,num;
#ifdef DEBUG
  if(S->num[SECT_FREE]>0){
    fprintf(stderr, "error: localSearchII() is called although there is a free vertex (num[SECT_FREE]=%d)\n", S->num[SECT_FREE]);
    exit(EXIT_FAILURE);
  }
#endif
  for(int r=0;r<S->num[SECT_TWO];r++){
    i = (T->a++)%S->num[SECT_TWO];
    u = S->Order[SECT_TWO][i];
    X[0] = NULL;
    X[1] = NULL;
    num = 0;
    // compute u's two solution neighbors, X[0] & X[1]
    // reset u's neighbors' counters
    for(auto itr=u->v->AL.begin(); itr!=u->v->AL.end(); ++itr){
      j = *itr;
      S->X[j]->counter = 0;
      if(S->X[j]->isSol){
	if(X[0]==NULL)
	  X[0] = S->X[j];
	else
	  X[1] = S->X[j];
      }
    }
    // reset X[1]'s neighbors' counters
    setNeighborsCounters(S, X[1], 0);
    // mark the neighbors of X[0] and X[1]
    for(bi=0;bi<2;bi++)
      for(auto itr=X[bi]->v->AL.begin(); itr!=X[bi]->v->AL.end(); ++itr){
	j = *itr;
	v = S->X[j];
	if(v==u)
	  continue;
	if(bi==0)
	  v->counter = 1;
	else
	  v->counter++;
	if(v->tightness==1)
	  num++;
	else if(v->tightness==2 && v->counter==2)
	  num++;
      }
    // search neighbor of u
    for(auto itr=u->v->AL.begin(); itr!=u->v->AL.end(); ++itr){
      j = *itr;
      v = S->X[j];
      if((v->tightness==1 || v->tightness==2)
	 && v->tightness==v->counter && v->isSol==false)
	num--;
    }
    if(num==0){
      for(bi=0;bi<2;bi++)
	drop(S, X[(T->a++)%2]->order);
      add(S, u->order);
      flag = true;
      break;
    }
  }
  return flag;
}


bool localSearchIII(ParamSet PS, Graph G, Solution S, Tool T){
  VertexProxy U[2],X[3],x,y,proxy;
  Vertex v;
  int i,j,k;

  /* I is used to scan 2- and 3-tight vertices simultaneously */
  int *I;
  int size_I;
  size_I = S->num[SECT_TWO]+S->num[SECT_MORE];
  I = new int[size_I];
  for(int r=0;r<size_I;r++)
    I[r] = r;
  shuffle(I, size_I, sizeof(int));

  /* T1N, T2N: 1- and 2-tight neighbors of solution vertices
     SolN: solution neighbors of 2-tight vertices */
  Seq *T1N,*T2N;
  int **SolN, prev_sect_two;
  T1N = new Seq[S->num[SECT_SOL]];  
  T2N = new Seq[S->num[SECT_SOL]];  
  SolN = new int*[S->num[SECT_TWO]];
  for(i=0;i<S->num[SECT_TWO];i++)
    SolN[i] = new int[2];
  prev_sect_two = S->num[SECT_TWO];

  /* construct T1N */
  for(i=0;i<S->num[SECT_SOL];i++){
    proxy = S->Order[SECT_SOL][i];
    v = proxy->v;
    for(int r=0;r<v->deg;r++){
      j = v->AL[ v->R[(T->a++)%v->deg] ];
      if(S->X[j]->tightness==1)
	T1N[i].push_back(j);
    }
  }
  
  /* construct SolN and T2N */
  for(i=0;i<S->num[SECT_TWO];i++){
    proxy = S->Order[SECT_TWO][i];
    v = proxy->v;
    k = 0;
    // collect the two solution neighbors of i-th 2-tight vertex
    for(int r=0;r<v->deg;r++){
      j = v->AL[ v->R[(T->a++)%v->deg] ];
      if(S->X[j]->isSol){
	SolN[i][k] = S->X[j]->order;
	T2N[S->X[j]->order].push_back(i);
	k++;
	if(k==2)
	  break;
      }
    }
  }

  bool impflag = false;
  
  for(int r=0;r<size_I;r++){
    if(I[r]>=S->num[SECT_TWO]){
      i = I[r]-S->num[SECT_TWO];
      U[0] = S->Order[SECT_MORE][i];
      if(U[0]->tightness!=3)
	continue;
      for(k=0;k<3;k++)
	X[k] = NULL;
      k = 0;
      for(int d=0;d<U[0]->v->deg;d++){
	j = U[0]->v->AL[ U[0]->v->R[(T->a++)%U[0]->v->deg] ];
	if(S->X[j]->isSol==false)
	  continue;
	X[k] = S->X[j];
	k++;
	if(k==3)
	  break;
      }
      if(localSearchIII_3TightBase(G, S, T, U[0], X)){
	impflag = true;
	break;
      }
    }
    else{
      i = I[r];
      Seq NX_2, NXP_3;
      BinTree NXP_2;
      U[0] = S->Order[SECT_TWO][i];
      for(int bi=0;bi<2;bi++){
	X[bi] = S->Order[SECT_SOL][SolN[i][bi]];
	setNeighborsCounters(S, X[bi], 0);
      }
      for(int bi=0;bi<2;bi++)
	for(auto itr=X[bi]->v->AL.begin(); itr!=X[bi]->v->AL.end(); ++itr){
	  k = *itr;
	  proxy = S->X[k];
	  proxy->counter++;
	  switch(proxy->tightness){
	  case 2:
	    if(proxy->counter==2)
	      NX_2.push_back(k);
	    else if(proxy->counter==1)
	      NXP_2[k] = 1;
	    break;
	  case 3:
	    if(proxy->counter==2)
	      NXP_3.push_back(k);
	    break;
	  }
	}
      for(auto itr=NX_2.begin(); itr!=NX_2.end(); ++itr){
	k = *itr;
	if(NXP_2.find(k)!=NXP_2.end())
	  NXP_2.erase(k);
      }
      sort(NXP_3.begin(), NXP_3.end());
    
      for(int bi=0;bi<2;bi++){
	for(auto itr=T2N[SolN[i][bi]].begin(); itr!=T2N[SolN[i][bi]].end(); ++itr){
	  bool adj=false;
	  j = *itr;
	  if(i==j)
	    continue;
	  U[1] = S->Order[SECT_TWO][j];
	  if(U[0]->v->AT.find(U[1]->v->id)!=U[0]->v->AT.end())
	    adj = true;
	  else if(U[0]->v->id >= U[1]->v->id)
	    continue;
	  x = S->Order[SECT_SOL][SolN[j][0]];
	  y = S->Order[SECT_SOL][SolN[j][1]];
	  if((X[0]==x || X[0]==y) && (X[1]==x || X[1]==y))
	    continue;
	  else if(X[0]==x || X[1]==x)
	    X[2] = y;
	  else if(X[0]==y || X[1]==y)
	    X[2] = x;
	  else{
	    fprintf(stderr, "error: something is happening at ls3. watch out.\n");
	    exit(EXIT_FAILURE);
	  }
	  if(localSearchIII_2TightBase(G, S, T, U, X, adj, T1N, NX_2, NXP_2, NXP_3)){
	    impflag = true;
	    break;
	  }	
	}
	if(impflag)
	  break;
      }
    }
    if(impflag)
      break;
  }
  
  delete[] I;
  for(int p=0;p<prev_sect_two;p++)
    delete[] SolN[p];
  delete[] SolN;
  delete[] T1N;
  delete[] T2N;
  return impflag;
}


bool localSearchIII_3TightBase(Graph G, Solution S, Tool T,
			       VertexProxy u, VertexProxy X[3]){
  VertexProxy v;
  Seq FD_seq, undom_seq;
  int i,j,tri,colored_num=0;  
  
  // collect F(D) as FD_seq
  setNeighborsCounters(S, u, 0);
  setNeighborsCounters(S, X[1], 0);
  setNeighborsCounters(S, X[2], 0);

  for(tri=0;tri<3;tri++)
    for(int r=0;r<X[tri]->v->deg;r++){
      i = X[tri]->v->AL[ X[tri]->v->R[(T->a++)%X[tri]->v->deg] ];
      v = S->X[i];
      if(v==u)
	continue;
      if(tri==0)
	v->counter = 1;
      else
	v->counter++;
      if(v->tightness==1)
	FD_seq.push_back(i);
      else if(v->tightness==2 && v->counter==2)
	FD_seq.push_back(i);
      else if(v->tightness==3 && v->counter==3)
	FD_seq.push_back(i);
    }
  
  // color vertices in N(u)
  updateColor(G, S, T);
  for(auto itr=u->v->AL.begin(); itr!=u->v->AL.end(); ++itr){
    i = *itr;
    v = S->X[i];
    v->color = T->color;
    if(v->tightness>=1 && v->tightness<=3 &&
       v->tightness==v->counter && v->isSol==false)
      colored_num++;
  }
  
  // improvement in easy case
  int FD_size;
  FD_size = FD_seq.size();
  if(colored_num >= FD_size-1){
    for(tri=0;tri<3;tri++)
      drop(S, X[(T->a++)%3]->order);
    add(S, u->order);
#ifdef DEBUG
    if(S->num[SECT_FREE]>1){
      fprintf(stderr, "error: the number of free vertices should be <=1, but it is %d\n", S->num[SECT_FREE]);
      exit(EXIT_FAILURE);
    }
#endif
    if(S->num[SECT_FREE]==1)
      add(S, S->Order[SECT_FREE][0]->order);
    return true;
  }

  // improvement in complicated case
  int prev_color;
  prev_color = T->color;
  for(int r=0;r<FD_size;r++){
    i = FD_seq[(T->a++)%FD_size];
    if(S->X[i]->color != prev_color)
      undom_seq.push_back(i);
  }
  updateColor(G, S, T);
  for(auto itr=undom_seq.begin(); itr!=undom_seq.end(); ++itr){
    i = *itr;
    S->X[i]->color = T->color;
  }  
  for(auto itr=undom_seq.begin(); itr!=undom_seq.end(); ++itr){
    int num = 0;
    i = *itr;
    for(auto itr2=S->X[i]->v->AL.begin(); itr2!=S->X[i]->v->AL.end(); ++itr2){
      j = *itr2;
      if(S->X[j]->color == T->color)
	num++;
    }
    int size = undom_seq.size();
    if(num==size-1){
      for(tri=0;tri<3;tri++)
	drop(S, X[(T->a++)%3]->order);
      add(S, u->order);
      add(S, S->X[i]->order);
      return true;
    }
  }
  return false;
}

bool localSearchIII_2TightBase(Graph G, Solution S, Tool T,
			       VertexProxy U[2], VertexProxy X[3], bool adj,
			       Seq *T1N, Seq &NX_2, BinTree &NXP_2, Seq &NXP_3){
  Seq NX;
  int i,j,Fsize=0;
  
  /* construct NX */
  // 2-tight
  getCommonSeqWithBT(NX, X[2]->v->AL, NXP_2);
  // 3-tight
  getCommonSeq(NX, X[2]->v->AL, NXP_3);

  updateColor(G, S, T);
  for(int tri=0;tri<3;tri++){
    j = X[tri]->order;
    for(auto itr=T1N[j].begin(); itr!=T1N[j].end(); ++itr){
      i = *itr;
      if(i==U[0]->v->id || i==U[1]->v->id)
	continue;
      if(S->X[i]->color != T->color)
	Fsize++;
      S->X[i]->color = T->color;
    }
  }
  for(auto itr=NX_2.begin(); itr!=NX_2.end(); ++itr){
    i = *itr;
    if(i==U[0]->v->id || i==U[1]->v->id)
      continue;
    if(S->X[i]->color != T->color)
      Fsize++;
    S->X[i]->color = T->color;
  }
  for(auto itr=NX.begin(); itr!=NX.end(); ++itr){
    i = *itr;
    if(i==U[0]->v->id || i==U[1]->v->id)
      continue;
    if(S->X[i]->color != T->color)
      Fsize++;
    S->X[i]->color = T->color;
  }
  
  /* improvement in easy case */
  if(adj==false){
    int covered_num=0;
    for(int bi=0;bi<2;bi++)
      for(auto itr=U[bi]->v->AL.begin(); itr!=U[bi]->v->AL.end(); ++itr){
	i = *itr;
	if(S->X[i]->color == T->color){
	  S->X[i]->color--;
	  covered_num++;
	}
      }
    if(covered_num!=Fsize)
      return false;
    for(int tri=0;tri<3;tri++)
      drop(S, X[(T->a++)%3]->order);
    for(int bi=0;bi<2;bi++)
      add(S, U[bi]->order);
    return true;
  }

  /* improvement in complicated case */
  int dom=0;
  for(auto itr=U[0]->v->AL.begin(); itr!=U[0]->v->AL.end(); ++itr){
    i = *itr;
    if(S->X[i]->color == T->color){
      S->X[i]->color--;
      dom++;
    }
  }
  int undom,num;
  undom = Fsize-dom;
  for(auto itr=T1N[X[2]->order].begin(); itr!=T1N[X[2]->order].end(); ++itr){
    i = *itr;
    if(S->X[i]->color != T->color)
      continue;    
    num = 0;    
    for(auto itr2=S->X[i]->v->AL.begin(); itr2!=S->X[i]->v->AL.end(); ++itr2){
      j = *itr2;
      if(S->X[j]->color == T->color)
	num++;
    }    
    if(num == undom-1){
      for(int tri=0;tri<3;tri++)
	drop(S, X[(T->a++)%3]->order);
      add(S, U[0]->order);
      add(S, S->X[i]->order);
      return true;
    }
  }
  return false;
}


void updatePenalty(ParamSet PS, Graph G, Solution S, Tool T){
  for(int i=0;i<S->num[SECT_SOL];i++){
    int k = S->Order[SECT_SOL][i]->v->id;
    T->penalty[k]++;
  }
  T->t_p++;
}


void forget(ParamSet PS, Graph G, Solution S, Tool T){
  if(T->t_p < PS->delay)
    return;
  T->t_p = 0;
  
  int f_half=PS->delay/2;
  for(int i=0;i<G->n;i++){
    T->penalty[i] = T->penalty[i]/2;
    if(T->penalty[i]>f_half)
      T->penalty[i] = f_half;
  }
}


void kick(ParamSet PS, Graph G, Solution Best, Solution S, Tool T){
  BinTree BT;
  Hash H;
  int i,j,k=1,r,s,minPenalty=INT_MAX;
  // if T->update=true, you don't need to copy Best to S
  if(T->update==false)
    copySolution(G, Best, S);
  
  // construct the hash of non-solution vertices
  for(s=SECT_FREE;s<SECTS;s++)
    for(r=0;r<S->num[s];r++){
      i = S->Order[s][r]->v->id;
      BT[i] = 1;
      if(T->penalty[i]<minPenalty)
	minPenalty = T->penalty[i];
    }
  
  // pick up (possibly) a few non-solution vertices
  while(1){
    vector<int> I;
    I.clear();
    for(auto itr=BT.begin(); itr!=BT.end(); itr++){
      if(k==1){
	if(T->penalty[itr->first]==minPenalty)
	  I.push_back(itr->first);
      }
      else
	I.push_back(itr->first);
    }
    if(k==1)
      i = I[(int)(genrand_real2() * I.size())];
    else{
      int i1,i2,i3;
      i1 = I[(int)(genrand_real2() * I.size())];
      i2 = I[(int)(genrand_real2() * I.size())];
      i3 = I[(int)(genrand_real2() * I.size())];
      i = i1;
      if(T->penalty[i2]<T->penalty[i])
	i = i2;
      if(T->penalty[i3]<T->penalty[i])
	i = i3;
    }
    BT.erase(i);
    H[i] = true;
    for(auto itr=S->X[i]->v->AL.begin(); itr!=S->X[i]->v->AL.end(); itr++){
      j = *itr;
      if(BT.find(j)!=BT.end())
	BT.erase(j);
    }
    // break if BT is empty
    if(BT.empty())
      break;
    // break with a probability 1/nu
    if(genrand_real2() < 1.0/(double)PS->nu)
      break;
    k++;
  }

  // forcibly addition
  for(auto itr=H.begin(); itr!=H.end(); ++itr){
    i = itr->first;
    for(auto itr2=S->X[i]->v->AL.begin(); itr2!=S->X[i]->v->AL.end(); ++itr2){
      j = *itr2;
      if(S->X[j]->isSol)
	drop(S, S->X[j]->order);
    }
    add(S, S->X[i]->order);
  }

  // add a free vertex to S repeatedly as long as one exists
  constructGreedySol(G, S);
}

