/********** 
  ILPS (Iterated Local & Plateau Search) for
  the Minimum Independent Dominating Set Problem

  init.cpp

  Copyright 2018 Kazuya Haraguchi
  Released Under the MIT License
  https://opensource.org/licenses/mit-license.php
**********/

#include "define.h"
#include "init.h"
#include "misc.h"


void outputUsage(FILE *out, int argc, char *argv[]){
  fprintf(out, "usage: %s [options]\n\n", argv[0]);
  fprintf(out, "options:\n");
  fprintf(out, "  -h (or -help) output this message.\n");
  fprintf(out, "  -seed=<INT>   random seed (%d)\n", INI_SEED);
  fprintf(out, "  -time=<INT>   time limit (%g)\n", INI_TIME);
  fprintf(out, "  -init=<INT>   initial solution (%d)\n", INI_INIT);
  fprintf(out, "                0:random, 1:greedy\n");
  fprintf(out, "  -nu=<INT>     exp # of added vertices in perturbation (%d)\n", INI_NU);
  fprintf(out, "  -itrmax=<INT> iteration times; if <0, it is set to (-itrmax)|V| (%d)\n", INI_ITRMAX);
  fprintf(out, "  -delay=<INT>  penalty delay (%d)\n", INI_DELAY);
  fprintf(out, "\n");
  fprintf(out, "Following the options, you are required to input the instance data.\n");
  fprintf(out, "Alternatively, you can feed the data by means of pipe; e.g., cat inst_file | %s\n\n", argv[0]);
}


void outputErrorMsg(char *msg){
  fprintf(stderr, "\nError: %s\n\n", msg);
  exit(EXIT_FAILURE);
} 


void readParam(int argc, char *argv[], ParamSet PS){
  char **S;
  int k,eqs;
  PS->seed = INI_SEED;
  PS->time = INI_TIME;
  PS->init = INI_INIT;
  PS->nu = INI_NU;
  PS->itrmax = INI_ITRMAX;
  PS->delay = INI_DELAY;
  for(k=1;k<argc;k++){
    eqs = getNumChar(argv[k], '=');
    if(argv[k][0] != '-' || eqs > 1)
      outputErrorMsg("invalid argument");
    S = split(argv[k]+1, '=');
    if(strcmp(S[0], "help") == 0 || strcmp(S[0], "h") == 0){
      outputUsage(stdout, argc, argv);
      exit(0);
    }
    else if(eqs != 1 || strlen(S[1]) == 0)
      outputErrorMsg("option value should be specified appropriately.");
    else if(strcmp(S[0], "seed") == 0)
      PS->seed = atoi(S[1]);
    else if(strcmp(S[0], "time") == 0)
      PS->time = atof(S[1]);
    else if(strcmp(S[0], "init") == 0)
      PS->init = atoi(S[1]);
    else if(strcmp(S[0], "nu") == 0)
      PS->nu = atoi(S[1]);
    else if(strcmp(S[0], "itrmax") == 0)
      PS->itrmax = atoi(S[1]);
    else if(strcmp(S[0], "delay") == 0)
      PS->delay = atoi(S[1]);
    else
      outputErrorMsg("illegal parameter name.");
    free(S[0]);
    if(eqs == 1)
      free(S[1]);
    free(S);
  }
}


#define fscan_next_ill(fp)\
  ({  long long int x=0;\
  int c;\
  c = fgetc(fp);\
  if(c!='\n' && c!='\t')x=anum[c];\
  while(' '!=(c=fgetc(fp))){\
  if(c=='\n')break;\
  if(c=='\t')break;\
  x=10*x+anum[c];\
  }\
x;\
  })


void readInstance(FILE *fp, Graph G){
  int i,j,k;
  int anum[58];
  
  anum[(int)'0']=0;//48
  anum[(int)'1']=1;
  anum[(int)'2']=2;
  anum[(int)'3']=3;
  anum[(int)'4']=4;
  anum[(int)'5']=5;
  anum[(int)'6']=6;
  anum[(int)'7']=7;
  anum[(int)'8']=8;
  anum[(int)'9']=9; //57
  anum[(int)' ']=0; //32
  
  G->m = fscan_next_ill(fp);
  G->n = fscan_next_ill(fp);

  // read vertex weights
  G->V = new Vertex[G->n];
  for(i=0;i<G->n;i++){
    G->V[i] = new struct _Vertex;
    G->V[i]->id = i;
    G->V[i]->AT.clear();
    G->V[i]->AL.clear();
    G->V[i]->w = fscan_next_ill(fp);
    if(G->V[i]->w != 1){
      fprintf(stderr, "error: current version requires that all vertex weights are 1.\n");
      exit(EXIT_FAILURE);
    }
  }
  
  // read edges
  for(k=0;k<G->m;k++){
    i = fscan_next_ill(fp);
    j = fscan_next_ill(fp);
    i--; j--; // in the assumed format, the index begins from 1
    G->V[i]->AT[j] = 1;
    G->V[j]->AT[i] = 1;
    G->V[i]->AL.push_back(j);
    G->V[j]->AL.push_back(i);
  }

  for(i=0;i<G->n;i++){
    std::sort(G->V[i]->AL.begin(), G->V[i]->AL.end());
    G->V[i]->deg = G->V[i]->AL.size();
    G->V[i]->R = new int[G->V[i]->deg];
    for(int r=0;r<G->V[i]->deg;r++)
      G->V[i]->R[r] = r;
    shuffle(G->V[i]->R, G->V[i]->deg, sizeof(int));
  }
}


void initSolution(Graph G, Solution S){
  int i,s;
  S->X = new VertexProxy[G->n];
  for(i=0;i<G->n;i++){
    S->X[i] = new struct _VertexProxy;
    S->X[i]->v = G->V[i];
    S->X[i]->sect = SECT_FREE;
    S->X[i]->order = i;
    S->X[i]->counter = 0;
    S->X[i]->color = INI_COLOR;
    S->X[i]->tightness = 0;
    S->X[i]->isSol = false;
  }
  for(s=0;s<SECTS;s++){
    S->Order[s] = new VertexProxy[G->n];
    S->num[s] = 0;
  }
  for(i=0;i<G->n;i++)
    S->Order[SECT_FREE][i] = S->X[i];
  S->num[SECT_FREE] = G->n;
}


void initTool(Graph G, Tool T){
  int i,tim;
  T->start_time = 0.;
  T->t = 1;
  T->itr_tb = 0;
  T->time_tb = 0.;
  T->penalty = new int[G->n];
  T->profit = new int[G->n];
  T->t_p = 0;
  for(i=0;i<G->n;i++)
    T->penalty[i] = 0;
  T->color = INI_COLOR+1;
  T->a = 0;
}


