/********** 
  ILPS (Iterated Local & Plateau Search) for
  the Minimum Independent Dominating Set Problem

  construct.app

  Copyright 2018 Kazuya Haraguchi
  Released Under the MIT License
  https://opensource.org/licenses/mit-license.php
**********/

#include "define.h"
#include "init.h"
#include "construct.h"
#include "misc.h"

#ifdef CONSTRUCT_MAIN

void outputUsageConst(FILE *out, int argc, char *argv[]);
void readParamConst(int argc, char *argv[], ParamSet PS);

int main(int argc, char *argv[]){
  ParamSet PS;
  Graph G;
  Solution S;
  bool firstflag = true;
  int i;
  
  /* preprocess */
  PS = new struct _ParamSet;
  G = new struct _Graph;
  S = new struct _Solution;

  readParamConst(argc, argv, PS);
  readInstance(stdin, G);
  initSolution(G, S);
  init_genrand(PS->seed);
  cpu_time();

  /* construct the initial solution */
  if(PS->init == 0)
    constructRandomSol(G, S);
  else
    constructGreedySol(G, S);

  /* output */
  for(i=0;i<G->n;i++){
    if(S->X[i]->isSol==false)
      continue;
    if(firstflag==false)
      printf(",");
    printf("%d", S->X[i]->v->id+1);
    firstflag = false;
  }
  printf("\n");
  return EXIT_SUCCESS;
}


void outputUsageConst(FILE *out, int argc, char *argv[]){
  fprintf(out, "usage: %s [options]\n\n", argv[0]);
  fprintf(out, "options:\n");
  fprintf(out, "  -h (or -help) output this message.\n");
  fprintf(out, "  -seed=<INT>   random seed (%d)\n", INI_SEED);
  fprintf(out, "  -init=<INT>   initial solution (%d)\n", INI_INIT);
  fprintf(out, "                0:random, 1:greedy\n");
  fprintf(out, "\n");
  fprintf(out, "Following the options, you are required to input the instance data.\n");
  fprintf(out, "Alternatively, you can feed the data by means of pipe; e.g., cat inst_file | %s\n\n", argv[0]);
}


void readParamConst(int argc, char *argv[], ParamSet PS){
  char **S;
  int k,eqs;
  PS->seed = INI_SEED;
  PS->init = INI_INIT;
  for(k=1;k<argc;k++){
    eqs = getNumChar(argv[k], '=');
    if(argv[k][0] != '-' || eqs > 1)
      outputErrorMsg("invalid argument");
    S = split(argv[k]+1, '=');
    if(strcmp(S[0], "help") == 0 || strcmp(S[0], "h") == 0){
      outputUsageConst(stdout, argc, argv);
      exit(0);
    }
    else if(eqs != 1 || strlen(S[1]) == 0)
      outputErrorMsg("option value should be specified appropriately.");
    else if(strcmp(S[0], "seed") == 0)
      PS->seed = atoi(S[1]);
    else if(strcmp(S[0], "init") == 0)
      PS->init = atoi(S[1]);
    else
      outputErrorMsg("illegal parameter name.");
    free(S[0]);
    if(eqs == 1)
      free(S[1]);
    free(S);
  }
}

#endif


int constructRandomSol(Graph G, Solution S){
  int i,size=0;
  while(S->num[SECT_FREE]>0){
    i = (int)(genrand_real2() * (double)S->num[SECT_FREE]);
    add(S, i);
    size++;
  }
  return size;
}


int constructGreedySol(Graph G, Solution S){
  VertexProxy x,y;
  int *M,sizeM,size=0,i,max,count;
  M = new int[G->n];
  while(S->num[SECT_FREE]>0){
    sizeM = 0;
    max = -1;
    for(i=0;i<S->num[SECT_FREE];i++){
      x = S->Order[SECT_FREE][i];
      count=0;
      for(auto itr=x->v->AL.begin(); itr!=x->v->AL.end(); ++itr){
	y = S->X[*itr];
	if(y->isSol==false && y->tightness==0)
	  count++;
      }
      if(count>max){
	M[0] = i;
	sizeM = 1;
	max = count;
      }
      else if(count==max){
	M[sizeM] = i;
	sizeM++;
      }
    }    
    i = (int)(genrand_real2() * (double)sizeM);
    add(S, M[i]);
    size++;
  }
  delete[] M;
  return size;
}

int constructGreedySolWithP(Graph G, Solution S, Tool T){
  VertexProxy x,y;
  vector<int> M;
  int size=0,i,k,max,count;
  while(S->num[SECT_FREE]>0){
    max = -1;
    for(i=0;i<S->num[SECT_FREE];i++){
      x = S->Order[SECT_FREE][i];
      count=0;
      for(auto itr=x->v->AL.begin(); itr!=x->v->AL.end(); ++itr){
	y = S->X[*itr];
	if(y->isSol==false && y->tightness==0)
	  count++;
      }
      if(count>max){
	M.clear();
	M.push_back(x->v->id);
	max = count;
      }
      else if(count==max)
	M.push_back(x->v->id);
    }
    k = getVertexWithLowestPenalty(M, T->penalty);
    add(S, S->X[M[k]]->order);
    size++;
  }
  return size;
}
